<?php
// Start output buffering to prevent any accidental output
ob_start();

// Start session first, before any output
if (session_status() === PHP_SESSION_NONE) {
    // Set session cookie path to current directory to work in any subfolder
    $session_path = dirname($_SERVER['SCRIPT_NAME']);
    if ($session_path === '/' || $session_path === '\\') {
        $session_path = '/';
    }
    // Set cookie params - use array syntax for PHP 7.3+, fallback for older versions
    if (version_compare(PHP_VERSION, '7.3.0', '>=')) {
        session_set_cookie_params([
            'lifetime' => 0,
            'path' => $session_path,
            'domain' => '',
            'secure' => false,
            'httponly' => true,
            'samesite' => 'Lax'
        ]);
    } else {
        session_set_cookie_params(0, $session_path, '', false, true);
    }
    session_start();
}

// Load config and util first
require_once dirname(__FILE__) . '/config.php';
require_once dirname(__FILE__) . '/util.php';

function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'ISO-8859-1'); }

// Handle login - process before headers
$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'login') {
    $email = trim($_POST['email'] ?? '');
    $pass = $_POST['password'] ?? '';
    
    // Debug mode - remove after fixing
    $debug_mode = isset($_GET['debug_login']);
    
    if (empty($email) || empty($pass)) {
        $error = 'Please fill all fields';
    } else {
        try {
            $pdo = db();
            
            // Check admin in database
            $stmt = $pdo->prepare("SELECT id, password_hash, active FROM counter_strike_admins WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            $admin = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($debug_mode) {
                ob_end_clean();
                echo "<pre>";
                echo "Email: " . htmlspecialchars($email) . "\n";
                echo "Password length: " . strlen($pass) . "\n";
                echo "Admin found: " . ($admin ? 'YES' : 'NO') . "\n";
                if ($admin) {
                    echo "Admin ID: " . $admin['id'] . "\n";
                    echo "Admin active: " . $admin['active'] . "\n";
                    echo "Password hash length: " . strlen($admin['password_hash']) . "\n";
                    echo "Password verify result: " . (password_verify($pass, $admin['password_hash']) ? 'TRUE' : 'FALSE') . "\n";
                }
                echo "</pre>";
                exit;
            }
            
            if ($admin) {
                if ($admin['active'] != 1) {
                    $error = 'Account is inactive';
                } elseif (password_verify($pass, $admin['password_hash'])) {
                    // Set session variables
                    $_SESSION['admin_logged_in'] = true;
                    $_SESSION['admin_id'] = intval($admin['id']);
                    $_SESSION['admin_email'] = $email;
                    
                    // Update last login
                    try {
                        $stmt = $pdo->prepare("UPDATE counter_strike_admins SET last_login = NOW() WHERE id = ?");
                        $stmt->execute([$admin['id']]);
                    } catch (Exception $e) {
                        // Ignore update error, login still works
                    }
                    
                    // Clear output buffer completely before redirect
                    if (ob_get_level() > 0) {
                        ob_end_clean();
                    }
                    
                    // Regenerate session ID for security (after setting session vars)
                    session_regenerate_id(true);
                    
                    // Redirect - use simple relative path (works in any folder)
                    header('Location: admin_add_skin.php', true, 302);
                    exit;
                } else {
                    $error = 'Invalid password';
                }
            } else {
                $error = 'Email not found in database';
            }
        } catch (Exception $e) {
            $error = 'Login error: ' . $e->getMessage();
        }
    }
}

// Now set headers after session and login processing
header('Content-Type: text/html; charset=windows-1252');
header('X-Frame-Options: ALLOWALL');
header('Content-Security-Policy: frame-ancestors *');

// Handle logout
if (isset($_GET['logout'])) {
    $_SESSION = array();
    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time()-3600, '/');
    }
    session_destroy();
    header('Location: admin_add_skin.php');
    exit;
}

// Check if logged in
$is_logged_in = isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;

// Debug mode - remove after testing
if (isset($_GET['debug'])) {
    echo "<pre>";
    echo "Session ID: " . session_id() . "\n";
    echo "Session status: " . (session_status() === PHP_SESSION_ACTIVE ? 'ACTIVE' : 'INACTIVE') . "\n";
    echo "Session data:\n";
    print_r($_SESSION);
    echo "\nIs logged in check: " . ($is_logged_in ? 'YES' : 'NO') . "\n";
    echo "admin_logged_in in session: " . (isset($_SESSION['admin_logged_in']) ? ($_SESSION['admin_logged_in'] ? 'TRUE' : 'FALSE') : 'NOT SET') . "\n";
    echo "Admin ID: " . ($_SESSION['admin_id'] ?? 'NOT SET') . "\n";
    echo "Admin email: " . ($_SESSION['admin_email'] ?? 'NOT SET') . "\n";
    echo "</pre>";
    exit;
}


// Handle add skin
$success_msg = '';
$error_msg = '';
if ($is_logged_in && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_skin') {
    $name = trim($_POST['name'] ?? '');
    $image_url = trim($_POST['image_url'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $description = trim($_POST['description'] ?? '');
    
    if ($name && $image_url && $price > 0) {
        try {
            $pdo = db();
            $stmt = $pdo->prepare("INSERT INTO counter_strike_skins (name, image_url, price, description) VALUES (?, ?, ?, ?)");
            $stmt->execute([$name, $image_url, $price, $description ?: null]);
            $success_msg = 'Skin added successfully!';
        } catch (Exception $e) {
            $error_msg = 'Error: ' . $e->getMessage();
        }
    } else {
        $error_msg = 'Please fill all required fields (name, image URL, price)';
    }
}

// Handle delete skin
if ($is_logged_in && isset($_GET['delete'])) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare("DELETE FROM counter_strike_skins WHERE id = ?");
        $stmt->execute([intval($_GET['delete'])]);
        $success_msg = 'Skin deleted successfully!';
    } catch (Exception $e) {
        $error_msg = 'Error: ' . $e->getMessage();
    }
}

// Get all skins
$skins = [];
if ($is_logged_in) {
    try {
        $pdo = db();
        $stmt = $pdo->query("SELECT * FROM counter_strike_skins ORDER BY created_at DESC");
        $skins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $error_msg = 'Error loading skins: ' . $e->getMessage();
    }
}

// Handle user management - add money
$user_management_msg = '';
$user_management_error = '';
$search_results = [];
$search_query = '';

if ($is_logged_in && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'add_money') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $amount = floatval($_POST['amount'] ?? 0);
        
        if ($user_id > 0 && $amount > 0) {
            try {
                $pdo = db();
                // Get current money
                $stmt = $pdo->prepare("SELECT id, money FROM counter_strike_users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($user_data) {
                    $new_money = floatval($user_data['money']) + $amount;
                    $stmt = $pdo->prepare("UPDATE counter_strike_users SET money = ? WHERE id = ?");
                    $stmt->execute([$new_money, $user_id]);
                    $user_management_msg = "Added {$amount} servercoins to user ID {$user_id}. New balance: {$new_money} servercoins.";
                } else {
                    $user_management_error = 'User not found';
                }
            } catch (Exception $e) {
                $user_management_error = 'Error: ' . $e->getMessage();
            }
        } else {
            $user_management_error = 'Invalid user ID or amount';
        }
    } elseif ($_POST['action'] === 'search_users') {
        $search_query = trim($_POST['search_query'] ?? '');
        if (!empty($search_query)) {
            try {
                $pdo = db();
                // Search by username or ID
                if (is_numeric($search_query)) {
                    $stmt = $pdo->prepare("SELECT id, username, money FROM counter_strike_users WHERE id = ? LIMIT 20");
                    $stmt->execute([intval($search_query)]);
                } else {
                    $stmt = $pdo->prepare("SELECT id, username, money FROM counter_strike_users WHERE username LIKE ? LIMIT 20");
                    $stmt->execute(['%' . $search_query . '%']);
                }
                $search_results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (Exception $e) {
                $user_management_error = 'Error searching: ' . $e->getMessage();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=windows-1252">
<title>Admin - Add Skins</title>
<style>
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

html {
    scroll-behavior: smooth;
}

body {
    background: #0b0b0b;
    color: #eee;
    font-family: Arial, Helvetica, sans-serif;
    margin: 0;
    padding: 20px;
}

.wrap {
    max-width: 800px;
    margin: 0 auto;
    padding: 20px;
    background: #111;
    border: 1px solid #222;
}

h3 {
    margin: 0 0 20px;
    text-align: center;
    color: #8bc34a;
    font-size: 24px;
    font-weight: bold;
    text-transform: uppercase;
}

.login-form, .add-form {
    margin-bottom: 30px;
    padding: 20px;
    background: #181818;
    border: 1px solid #333;
}

label {
    display: block;
    margin: 10px 0 5px;
    color: #ccc;
    font-size: 14px;
    font-weight: bold;
}

input[type="text"],
input[type="email"],
input[type="password"],
input[type="number"],
textarea {
    width: 100%;
    padding: 10px;
    background: #0b0b0b;
    border: 2px solid #8bc34a;
    color: #eee;
    font-size: 14px;
    font-family: Arial, sans-serif;
}

input:focus, textarea:focus {
    border-color: #4caf50;
    outline: none;
}

button {
    padding: 12px 24px;
    background: #8bc34a;
    border: 2px solid #4caf50;
    color: #fff;
    font-size: 14px;
    font-weight: bold;
    cursor: pointer;
    text-transform: uppercase;
    margin-top: 15px;
    width: 100%;
}

button:hover {
    background: #4caf50;
}

.msg {
    padding: 12px;
    margin: 15px 0;
    border-radius: 4px;
    text-align: center;
    font-weight: bold;
}

.msg.success {
    background: rgba(139, 195, 74, 0.2);
    border: 1px solid #8bc34a;
    color: #8bc34a;
}

.msg.error {
    background: rgba(244, 67, 54, 0.2);
    border: 1px solid #f44336;
    color: #f44336;
}

.skins-list {
    margin-top: 30px;
}

.skin-item {
    display: flex;
    align-items: center;
    padding: 15px;
    margin-bottom: 10px;
    background: #181818;
    border: 1px solid #333;
}

.skin-item img {
    width: 80px;
    height: 80px;
    object-fit: cover;
    border: 2px solid #8bc34a;
    margin-right: 15px;
}

.skin-info {
    flex: 1;
}

.skin-name {
    color: #8bc34a;
    font-weight: bold;
    font-size: 16px;
    margin-bottom: 5px;
}

.skin-price {
    color: #4caf50;
    font-size: 14px;
}

.skin-actions {
    margin-left: 15px;
}

.btn-delete {
    padding: 8px 16px;
    background: #f44336;
    border: 2px solid #d32f2f;
    color: #fff;
    font-size: 12px;
    width: auto;
    margin: 0;
}

.btn-delete:hover {
    background: #d32f2f;
}

.logout-link {
    display: inline-block;
    margin-top: 20px;
    color: #8bc34a;
    text-decoration: none;
    font-size: 14px;
}

.logout-link:hover {
    color: #4caf50;
    text-decoration: underline;
}
</style>
</head>
<body>
<div class="wrap">
  <h3>Admin - Add Skins</h3>
  
  <?php if (!$is_logged_in): ?>
    <!-- Login Form -->
    <div class="login-form">
      <?php if (isset($error)): ?>
        <div class="msg error"><?= h($error) ?></div>
      <?php endif; ?>
      
      <form method="post">
        <input type="hidden" name="action" value="login">
        <label>Email:</label>
        <input type="email" name="email" required autocomplete="email" placeholder="your@email.com">
        <label>Password:</label>
        <input type="password" name="password" required autocomplete="current-password" placeholder="Your password">
        <button type="submit">Login</button>
      </form>
      <a href="admin_register.php" class="login-link" style="display: block; text-align: center; margin-top: 15px; color: #8bc34a; text-decoration: none; font-size: 14px;">Don't have an account? Register</a>
    </div>
  <?php else: ?>
    <!-- Add Skin Form -->
    <div class="add-form">
      <?php if ($success_msg): ?>
        <div class="msg success"><?= h($success_msg) ?></div>
      <?php endif; ?>
      <?php if ($error_msg): ?>
        <div class="msg error"><?= h($error_msg) ?></div>
      <?php endif; ?>
      
      <form method="post">
        <input type="hidden" name="action" value="add_skin">
        <label>Skin Name:</label>
        <input type="text" name="name" required placeholder="e.g., AK-47 Redline">
        <label>Image URL:</label>
        <input type="text" name="image_url" required placeholder="https://example.com/image.png">
        <label>Price:</label>
        <input type="number" name="price" step="0.01" min="0" required placeholder="0.00">
        <label>Description (optional):</label>
        <textarea name="description" rows="3" placeholder="Skin description..."></textarea>
        <button type="submit">Add Skin</button>
      </form>
    </div>
    
    <!-- Existing Skins List -->
    <div class="skins-list">
      <h3 style="margin-bottom: 15px; font-size: 20px;">Existing Skins</h3>
      <?php if (empty($skins)): ?>
        <p style="text-align: center; color: #666;">No skins added yet.</p>
      <?php else: ?>
        <?php foreach ($skins as $skin): ?>
          <div class="skin-item">
            <img src="<?= h($skin['image_url']) ?>" alt="<?= h($skin['name']) ?>" onerror="this.src='data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'80\' height=\'80\'%3E%3Crect fill=\'%23333\' width=\'80\' height=\'80\'/%3E%3Ctext x=\'50%25\' y=\'50%25\' text-anchor=\'middle\' dy=\'.3em\' fill=\'%23666\' font-size=\'12\'%3ENo Image%3C/text%3E%3C/svg%3E'">
            <div class="skin-info">
              <div class="skin-name"><?= h($skin['name']) ?></div>
              <div class="skin-price">Price: $<?= number_format($skin['price'], 2) ?></div>
              <?php if ($skin['description']): ?>
                <div style="color: #999; font-size: 12px; margin-top: 5px;"><?= h($skin['description']) ?></div>
              <?php endif; ?>
            </div>
            <div class="skin-actions">
              <a href="?delete=<?= $skin['id'] ?>" class="btn-delete" onclick="return confirm('Are you sure you want to delete this skin?')">Delete</a>
            </div>
          </div>
        <?php endforeach; ?>
      <?php endif; ?>
    </div>
    
    <!-- User Management Section -->
    <div class="add-form" style="margin-top: 40px;">
      <h3 style="margin-bottom: 20px; font-size: 20px;">User Management - Add Servercoins</h3>
      
      <?php if ($user_management_msg): ?>
        <div class="msg success"><?= h($user_management_msg) ?></div>
      <?php endif; ?>
      <?php if ($user_management_error): ?>
        <div class="msg error"><?= h($user_management_error) ?></div>
      <?php endif; ?>
      
      <!-- Search Users -->
      <div style="margin-bottom: 20px;">
        <form method="post">
          <input type="hidden" name="action" value="search_users">
          <label>Search User (by username or ID):</label>
          <input type="text" name="search_query" value="<?= h($search_query) ?>" placeholder="Enter username or user ID" required>
          <button type="submit" style="margin-top: 10px;">Search Users</button>
        </form>
      </div>
      
      <!-- Search Results -->
      <?php if (!empty($search_results)): ?>
        <div style="margin-top: 20px;">
          <h4 style="color: #8bc34a; margin-bottom: 15px;">Search Results:</h4>
          <?php foreach ($search_results as $user): ?>
            <div class="skin-item" style="margin-bottom: 15px;">
              <div class="skin-info" style="flex: 1;">
                <div class="skin-name">ID: <?= h($user['id']) ?> | Username: <?= h($user['username']) ?></div>
                <div class="skin-price">Current Balance: <?= number_format(floatval($user['money']), 2) ?> servercoins</div>
              </div>
              <div class="skin-actions">
                <form method="post" style="display: inline-block;">
                  <input type="hidden" name="action" value="add_money">
                  <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                  <input type="number" name="amount" step="0.01" min="0.01" placeholder="Amount" required style="width: 120px; padding: 8px; margin-right: 10px; background: #0b0b0b; border: 2px solid #8bc34a; color: #eee;">
                  <button type="submit" class="btn-delete" style="background: #8bc34a; border-color: #4caf50; width: auto; margin: 0;">Add Servercoins</button>
                </form>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      <?php elseif (!empty($search_query)): ?>
        <div style="text-align: center; color: #666; padding: 20px;">
          No users found matching "<?= h($search_query) ?>"
        </div>
      <?php endif; ?>
    </div>
    
    <a href="?logout=1" class="logout-link">Logout</a>
  <?php endif; ?>
</div>
</body>
</html>
<?php
// End output buffering and send output
ob_end_flush();
?>

